<?php
/**
 * Invoice Ninja (https://invoiceninja.com).
 *
 * @link https://github.com/invoiceninja/invoiceninja source repository
 *
 * @copyright Copyright (c) 2021. Invoice Ninja LLC (https://invoiceninja.com)
 *
 * @license https://www.elastic.co/licensing/elastic-license
 */

namespace App\Filters;

use App\Models\RecurringExpense;
use App\Models\User;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;

/**
 * RecurringExpenseFilters.
 */
class RecurringExpenseFilters extends QueryFilters
{
    /**
     * Filter based on search text.
     *
     * @param string query filter
     * @return Builder
     * @deprecated
     */
    public function filter(string $filter = '') : Builder
    {
        if (strlen($filter) == 0) {
            return $this->builder;
        }

        return  $this->builder->where(function ($query) use ($filter) {
            $query->where('recurring_expenses.name', 'like', '%'.$filter.'%')
                          ->orWhere('recurring_expenses.id_number', 'like', '%'.$filter.'%')
                          ->orWhere('recurring_expenses.custom_value1', 'like', '%'.$filter.'%')
                          ->orWhere('recurring_expenses.custom_value2', 'like', '%'.$filter.'%')
                          ->orWhere('recurring_expenses.custom_value3', 'like', '%'.$filter.'%')
                          ->orWhere('recurring_expenses.custom_value4', 'like', '%'.$filter.'%');
        });
    }

    /**
     * Filters the list based on the status
     * archived, active, deleted.
     *
     * @param string filter
     * @return Builder
     */
    public function status(string $filter = '') : Builder
    {
        if (strlen($filter) == 0) {
            return $this->builder;
        }

        $table = 'expenses';
        $filters = explode(',', $filter);

        return $this->builder->where(function ($query) use ($filters, $table) {
            $query->whereNull($table.'.id');

            if (in_array(parent::STATUS_ACTIVE, $filters)) {
                $query->orWhereNull($table.'.deleted_at');
            }

            if (in_array(parent::STATUS_ARCHIVED, $filters)) {
                $query->orWhere(function ($query) use ($table) {
                    $query->whereNotNull($table.'.deleted_at');

                    if (! in_array($table, ['users'])) {
                        $query->where($table.'.is_deleted', '=', 0);
                    }
                });
            }

            if (in_array(parent::STATUS_DELETED, $filters)) {
                $query->orWhere($table.'.is_deleted', '=', 1);
            }
        });
    }

    /**
     * Sorts the list based on $sort.
     *
     * @param string sort formatted as column|asc
     * @return Builder
     */
    public function sort(string $sort) : Builder
    {
        $sort_col = explode('|', $sort);

        return $this->builder->orderBy($sort_col[0], $sort_col[1]);
    }

    /**
     * Returns the base query.
     *
     * @param int company_id
     * @param User $user
     * @return Builder
     * @deprecated
     */
    public function baseQuery(int $company_id, User $user) : Builder
    {
        $query = DB::table('recurring_expenses')
            ->join('companies', 'companies.id', '=', 'recurring_expenses.company_id')
            ->where('recurring_expenses.company_id', '=', $company_id)
            ->select(
                DB::raw('COALESCE(recurring_expenses.country_id, companies.country_id) country_id'),
                'recurring_expenses.id',
                'recurring_expenses.private_notes',
                'recurring_expenses.custom_value1',
                'recurring_expenses.custom_value2',
                'recurring_expenses.custom_value3',
                'recurring_expenses.custom_value4',
                'recurring_expenses.created_at',
                'recurring_expenses.created_at as expense_created_at',
                'recurring_expenses.deleted_at',
                'recurring_expenses.is_deleted',
                'recurring_expenses.user_id',
            );

        /*
         * If the user does not have permissions to view all invoices
         * limit the user to only the invoices they have created
         */
        if (Gate::denies('view-list', RecurringExpense::class)) {
            $query->where('recurring_expenses.user_id', '=', $user->id);
        }

        return $query;
    }

    /**
     * Filters the query by the users company ID.
     *
     * @return Illuminate\Database\Query\Builder
     */
    public function entityFilter()
    {
        return $this->builder->company();
    }
}
